<?php

import('libs/plugins/validator.php');

/**
 * キューの取得
 *
 * @param array $queries
 * @param array $options
 *
 * @return array
 */
function select_queues($queries, $options = array())
{
    $queries = db_placeholder($queries);

    // キューを取得
    $queries['from'] = DATABASE_PREFIX . 'queues';

    // 削除済みデータは取得しない
    if (!isset($queries['where'])) {
        $queries['where'] = 'TRUE';
    }
    $queries['where'] = 'deleted IS NULL AND (' . $queries['where'] . ')';

    // データを取得
    return db_select($queries);
}

/**
 * キューの登録
 *
 * @param array $queries
 * @param array $options
 *
 * @return resource
 */
function insert_queues($queries, $options = array())
{
    $queries = db_placeholder($queries);

    // 初期値を取得
    $defaults = default_queues();

    if (isset($queries['values']['created'])) {
        if ($queries['values']['created'] === false) {
            unset($queries['values']['created']);
        }
    } else {
        $queries['values']['created'] = $defaults['created'];
    }
    if (isset($queries['values']['modified'])) {
        if ($queries['values']['modified'] === false) {
            unset($queries['values']['modified']);
        }
    } else {
        $queries['values']['modified'] = $defaults['modified'];
    }

    // データを登録
    $queries['insert_into'] = DATABASE_PREFIX . 'queues';

    return db_insert($queries);
}

/**
 * キューの編集
 *
 * @param array $queries
 * @param array $options
 *
 * @return resource
 */
function update_queues($queries, $options = array())
{
    $queries = db_placeholder($queries);

    // 初期値を取得
    $defaults = default_queues();

    if (isset($queries['set']['modified'])) {
        if ($queries['set']['modified'] === false) {
            unset($queries['set']['modified']);
        }
    } else {
        $queries['set']['modified'] = $defaults['modified'];
    }

    // データを編集
    $queries['update'] = DATABASE_PREFIX . 'queues';

    return db_update($queries);
}

/**
 * キューの削除
 *
 * @param array $queries
 * @param array $options
 *
 * @return resource
 */
function delete_queues($queries, $options = array())
{
    $queries = db_placeholder($queries);
    $options = array(
        'softdelete' => isset($options['softdelete']) ? $options['softdelete'] : true,
    );

    if ($options['softdelete'] === true) {
        // データを編集
        $resource = db_update(array(
            'update' => DATABASE_PREFIX . 'queues AS queues',
            'set'    => array(
                'deleted' => localdate('Y-m-d H:i:s'),
            ),
            'where'  => isset($queries['where']) ? $queries['where'] : '',
            'limit'  => isset($queries['limit']) ? $queries['limit'] : '',
        ));
        if (!$resource) {
            return $resource;
        }
    } else {
        // データを削除
        $resource = db_delete(array(
            'delete_from' => DATABASE_PREFIX . 'queues AS queues',
            'where'       => isset($queries['where']) ? $queries['where'] : '',
            'limit'       => isset($queries['limit']) ? $queries['limit'] : '',
        ));
        if (!$resource) {
            return $resource;
        }
    }

    return $resource;
}

/**
 * キューの正規化
 *
 * @param array $queries
 * @param array $options
 *
 * @return array
 */
function normalize_queues($queries, $options = array())
{
    // 送信開始日時
    if (isset($queries['started'])) {
        $queries['started'] = mb_convert_kana($queries['started'], 'n', MAIN_INTERNAL_ENCODING);
    }

    // 送信完了日時
    if (isset($queries['completed'])) {
        $queries['completed'] = mb_convert_kana($queries['completed'], 'n', MAIN_INTERNAL_ENCODING);
    }

    // 送信失敗回数
    if (isset($queries['failed'])) {
        $queries['failed'] = mb_convert_kana($queries['failed'], 'n', MAIN_INTERNAL_ENCODING);
    }

    return $queries;
}

/**
 * キューの検証
 *
 * @param array $queries
 * @param array $options
 *
 * @return array
 */
function validate_queues($queries, $options = array())
{
    $messages = array();

    // 送信開始日時
    if (isset($queries['started'])) {
        if (!validator_datetime($queries['started'])) {
            $messages['started'] = '送信開始日時の書式が不正です。';
        }
    }

    // 送信完了日時
    if (isset($queries['completed'])) {
        if (!validator_datetime($queries['completed'])) {
            $messages['completed'] = '送信完了日時の書式が不正です。';
        }
    }

    // 送信失敗回数
    if (isset($queries['failed'])) {
        if (!validator_required($queries['failed'])) {
            $messages['failed'] = '送信失敗回数が入力されていません。';
        } elseif (!validator_numeric($queries['failed'])) {
            $messages['failed'] = '送信失敗回数は半角数字で入力してください。';
        }
    }

    // 送信先
    if (isset($queries['mailto'])) {
        if (!validator_required($queries['mailto'])) {
            $messages['mailto'] = '送信先が入力されていません。';
        }
    }

    // 件名
    if (isset($queries['subject'])) {
        if (!validator_required($queries['subject'])) {
            $messages['subject'] = '件名が入力されていません。';
        } elseif (!validator_max_length($queries['subject'], 80)) {
            $messages['subject'] = '件名は80文字以内で入力してください。';
        }
    }

    // 本文
    if (isset($queries['message'])) {
        if (!validator_required($queries['message'])) {
            $messages['message'] = '本文が入力されていません。';
        }
    }

    return $messages;
}

/**
 * キューの初期値
 *
 * @return array
 */
function default_queues()
{
    return array(
        'id'        => null,
        'created'   => localdate('Y-m-d H:i:s'),
        'modified'  => localdate('Y-m-d H:i:s'),
        'deleted'   => null,
        'started'   => null,
        'completed' => null,
        'failed'    => 0,
        'mailto'    => '',
        'subject'   => '',
        'message'   => '',
        'header'    => null,
        'parameter' => null,
    );
}
